package de.ullisroboterseite.ursai2sensor;

import com.google.appinventor.components.common.Sensitivity;
import com.google.appinventor.components.runtime.util.OrientationSensorUtil;
import com.google.appinventor.components.runtime.util.SdkLevel;
import com.google.appinventor.components.runtime.util.YailList;
import com.google.appinventor.components.runtime.util.OrientationSensorUtil;
import com.google.appinventor.components.runtime.util.FroyoUtil;

import android.content.res.Configuration;
import android.content.res.Resources;
import android.os.Build;
import android.os.Handler;
import android.content.Context;
import android.hardware.Sensor;
import android.hardware.SensorEvent;

import android.util.Log;
import android.view.Surface;
import android.view.WindowManager;
import android.view.Display;

import android.hardware.SensorEvent;
import android.hardware.SensorEventListener;
import android.hardware.SensorManager;

import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import java.util.Queue;

public class OrientationSensor extends SensorBase {
    final Sensor magneticFieldSensor; // Reference to Android sensor instance

    // offsets in array returned by SensorManager.getOrientation()
    private static final int AZIMUTH = 0;
    private static final int PITCH = 1;
    private static final int ROLL = 2;
    private static final int DIMENSIONS = 3; // Warning: specific to our universe

    // Pre-allocated arrays to hold sensor data so that we don't cause so many garbage collections
    // while processing sensor events. All are used only in onSensorChanged.
    private final float[] accels = new float[DIMENSIONS]; // acceleration vector
    private final float[] mags = new float[DIMENSIONS]; // magnetic field vector

    // Flags to tell whether the above arrays are filled. They are set in onSensorChanged and cleared
    // in stopListening.
    private boolean accelsFilled;
    private boolean magsFilled;

    // Pre-allocated matrixes used to compute orientation values from acceleration and magnetic
    // field data.
    private final float[] rotationMatrix = new float[DIMENSIONS * DIMENSIONS];
    private final float[] inclinationMatrix = new float[DIMENSIONS * DIMENSIONS];
    private final float[] values = new float[DIMENSIONS];

    // Used to launch Runnables on the UI Thread after a delay
    //   final Handler androidUIHandler;

    public OrientationSensor(UrsAI2Sensor ursAI2Sensor) {
        super(ursAI2Sensor, SensorType.OrientationSensor, Sensor.TYPE_ACCELEROMETER);
        magneticFieldSensor = sensorManager.getDefaultSensor(Sensor.TYPE_MAGNETIC_FIELD);
    }

    /*
    * Returns the rotation of the screen from its "natural" orientation.
    * Note that this is the angle of rotation of the drawn graphics on the
    * screen, which is the opposite direction of the physical rotation of the
    * device. For example, if the device is rotated 90 degrees counter-clockwise,
    * to compensate rendering will be rotated by 90 degrees clockwise and thus
    * the returned value here will be Surface.ROTATION_90.  Return values will
    * be in the set Surface.ROTATION_{0,90,180,270}.
    */
    public int getScreenRotation() {
        Display display = ((WindowManager) UrsAI2Sensor.thisInstance.thisForm.getSystemService(Context.WINDOW_SERVICE))
                .getDefaultDisplay();
        if (SdkLevel.getLevel() >= SdkLevel.LEVEL_FROYO) {
            return FroyoUtil.getRotation(display);
        } else {
            return display.getOrientation();
        }
    }

    // Indicates whether this type of sensor is available
    @Override
    public boolean getAvailable() {
        return androidSensor != null && magneticFieldSensor != null;
    }

    // The minimum delay allowed between two events in microseconds
    // Can be overridden by the derived class
    @Override
    public int getMinDelay() {
        if (getAvailable())
            // return the bigger value
            return Math.max(androidSensor.getMinDelay(), magneticFieldSensor.getMinDelay());
        else
            return -1;
    }

    // The maximum delay allowed between two events in microseconds
    // Can be overridden by the derived class
    @Override
    public int getMaxDelay() {
        if (getAvailable())
            // return the smaller value
            return Math.min(androidSensor.getMaxDelay(), magneticFieldSensor.getMaxDelay());
        else
            return -1;
    }

    // Maximum range of the sensor in the sensor's unit.
    // Meaningless
    public float getMaximumRange() {
        return -1;
    }

    // Resolution of the sensor in the sensor's unit.
    // Meaningless
    public float getResolution() {
        return -1;
    }

    @Override
    public void onStart(int argDelay, YailList control) {
        // Acceleromter is started in SensorBase.Start.
        sensorManager.registerListener(sensorListener, magneticFieldSensor, argDelay);
    }

    @Override
    public void onStop() {
        // Acceleromter is stopped in SensorBase.Stop
        sensorManager.unregisterListener(sensorListener, magneticFieldSensor);
    }

    // Check whether the sensor data is intended for this sensor.
    boolean acceptSensorData(SensorEvent sensorEvent) {
        return sensorEvent.sensor == androidSensor || sensorEvent.sensor == magneticFieldSensor;
    }

    // SensorListener implementation
    @Override
    public void onSensorChanged(SensorEvent sensorEvent) {
        float azimuth; // degrees
        float pitch; // degrees
        float roll; // degrees
        int accuracy = 0;

        int eventType = sensorEvent.sensor.getType();

        // Save the new sensor information about acceleration or the magnetic field.
        switch (eventType) {
            case Sensor.TYPE_ACCELEROMETER:
                // Update acceleration array.
                System.arraycopy(sensorEvent.values, 0, accels, 0, DIMENSIONS);
                accelsFilled = true;
                // Only update the accuracy property for the accelerometer.
                accuracy = sensorEvent.accuracy;
                return; // Event is triggered from magnetic field sensor

            case Sensor.TYPE_MAGNETIC_FIELD:
                // Update magnetic field array.
                System.arraycopy(sensorEvent.values, 0, mags, 0, DIMENSIONS);
                magsFilled = true;
                break;

            default:
                Log.e(LOG_TAG, "Unexpected sensor type: " + eventType);
                return;
        }

        // If we have both acceleration and magnetic information, recompute values.
        if (accelsFilled && magsFilled) {
            SensorManager.getRotationMatrix(rotationMatrix, // output
                    inclinationMatrix, // output
                    accels,
                    mags);
            SensorManager.getOrientation(rotationMatrix, values);

            // Make sure values are in expected range.
            azimuth = OrientationSensorUtil.normalizeAzimuth(
                    (float) Math.toDegrees(values[AZIMUTH]));
            pitch = OrientationSensorUtil.normalizePitch(
                    (float) Math.toDegrees(values[PITCH]));
            // Sign change for roll is for compatibility with earlier versions
            // of App Inventor that got orientation sensor information differently.
            roll = OrientationSensorUtil.normalizeRoll(
                    (float) -Math.toDegrees(values[ROLL]));

            // Adjust pitch and roll for phone rotation (e.g., landscape)
            int rotation = getScreenRotation();
            switch (rotation) {
                case Surface.ROTATION_0: // normal rotation
                    break;
                case Surface.ROTATION_90: // phone is turned 90 degrees counter-clockwise
                    float temp = -pitch;
                    pitch = -roll;
                    roll = temp;
                    break;
                case Surface.ROTATION_180: // phone is rotated 180 degrees
                    roll = -roll;
                    break;
                case Surface.ROTATION_270: // phone is turned 90 degrees clockwise
                    temp = pitch;
                    pitch = roll;
                    roll = temp;
                    break;
                default:
                    Log.e(LOG_TAG, "Illegal value for getScreenRotation(): " + rotation);
                    break;
            }

            // Raise event.
            List<Float> valueList = new ArrayList<>();
            valueList.add(azimuth);
            valueList.add(pitch);
            valueList.add(roll);
            reportSensorData(sensorType, valueList, accuracy);
        }
    }
}