package de.ullisroboterseite.ursai2sensor;

import com.google.appinventor.components.common.Sensitivity;
import com.google.appinventor.components.runtime.util.YailList;

import android.hardware.Sensor;
import android.hardware.SensorEvent;
import android.hardware.SensorEventListener;
import android.hardware.SensorManager;
import android.content.Context;
import android.util.Log;

import java.util.ArrayList;
import java.util.List;

// Base for all sensors
public abstract class SensorBase implements SensorEventListener {
    static final public String LOG_TAG = UrsAI2Sensor.LOG_TAG;
    UrsAI2Sensor ursAI2Sensor; // Instance of the extension, e.g. for triggering events.
    SensorListener sensorListener; // Listener für sensor events

    final public SensorManager sensorManager; // https://developer.android.com/reference/android/hardware/SensorManager
    final public Sensor androidSensor; // Reference to Android sensor instance
    final public SensorType sensorType; // Type of this sensor (enum value)

    List<Float> sensorDataList = new ArrayList<>(); // List of sensor data

    int delay; // Time interval between two sensor events

    boolean hasReadings = false; // Indicates whether there are valid measured values

    // Initializes a new instance of class SensorBase
    // ====================================================================================
    // ursAI2Sensor:      Reference to the instance of the extensions
    // sensorType:        Type of this sensor, e.g. SensorType.Accelerometer
    // androidSensorType: Android sensor type, e.g. Sensor.TYPE_ACCELEROMETER
    // ursAI2Sensor:      Check whether the standard implementation of 'hasValidData'
    //                    is appropriate for the sensor.
    // ====================================================================================
    static boolean logList = true;

    public SensorBase(UrsAI2Sensor ursAI2Sensor, SensorType sensorType, int androidSensorType) {
        this.ursAI2Sensor = ursAI2Sensor;
        this.sensorType = sensorType;
        sensorListener = new SensorListener(this);

        sensorManager = (SensorManager) ursAI2Sensor.thisForm.getSystemService(Context.SENSOR_SERVICE);

        androidSensor = sensorManager.getDefaultSensor(androidSensorType);

        if (logList) {
            logList = false;
            List<Sensor> sl = sensorManager.getSensorList(Sensor.TYPE_ALL);
            for (Sensor sensor : sl) {
                Log.d(LOG_TAG, sensor.getType() + " / " + sensor.getStringType() + " / " + sensor.getName());
            }
        }

    }

    // Is called before the sensor is registered to the Sensormanager for special use of Sensivity and delay.
    public abstract void onStart(int delay, YailList control);

    // Is called after the sensor is unregistered from he Sensormanager.
    public abstract void onStop();

    // The minimum delay allowed between two events in microseconds
    // Can be overridden by the derived class
    public int getMinDelay() {
        if (androidSensor != null)
            return androidSensor.getMinDelay();
        else
            return -1;
    }

    // The maximum delay allowed between two events in microseconds
    // Can be overridden by the derived class
    public int getMaxDelay() {
        if (androidSensor != null)
            return androidSensor.getMaxDelay();
        else
            return -1;
    }

    // Indicates whether this type of sensor is available
    public boolean getAvailable() {
        return androidSensor != null;
    }

    // Delivers the most recently determined measured values
    public YailList getValues() {
          return YailList.makeList(sensorDataList);
    }

    // Indicates whether there are valid measured values
    // Can be overridden by derived classes if other conditions apply
    public boolean hasValidData() {
        return hasReadings;
    }

    // Maximum range of the sensor in the sensor's unit.
    public float getMaximumRange() {
        if (androidSensor != null)
            return androidSensor.getMaximumRange();
        else
            return -1;
    }

    // Resolution of the sensor in the sensor's unit.
    public float getResolution() {
        if (androidSensor != null)
            return androidSensor.getResolution();
        else
            return -1;
    }

    // Start getting readings for this sensor
    public void start(int argDelay, YailList control) {
        if (androidSensor == null)
            return;
        try {
            delay = argDelay;
            onStart(delay, control);
            if (delay < androidSensor.getMinDelay())
                delay = androidSensor.getMinDelay();
            if (delay > androidSensor.getMaxDelay())
                delay = androidSensor.getMaxDelay();
            Log.d(LOG_TAG, "Start: " + sensorType + " delay: " + delay);

            hasReadings = false;
            sensorManager.registerListener(sensorListener, androidSensor, delay); // delay may be modified
        } catch (Exception e) {
            DebugUtil.LogExecption(e);
        }

    }

    // Stops getting readings for this sensor
    public void stop() {
        if (androidSensor == null)
            return;

        Log.d(LOG_TAG, "Stop: " + sensorType);
        hasReadings = false;
        sensorManager.unregisterListener(sensorListener, androidSensor);
        onStop();
    }

    // SensorListener implementation
    @Override
    public void onAccuracyChanged(Sensor sensor, int accuracy) {
    }

    // Check whether the sensor data is intended for this sensor.
    boolean acceptSensorData(SensorEvent sensorEvent) {
        return sensorEvent.sensor == androidSensor;
    }

    // Handles new sensor data
    public void newSensorData(SensorEvent sensorEvent) {
        if (acceptSensorData(sensorEvent)) {
            onSensorChanged(sensorEvent);
            hasReadings = true;
        }
    }

    @Override
    public abstract void onSensorChanged(SensorEvent sensorEvent);

    public void reportSensorData(SensorType sensorType, List<Float> data, int accuracy) {
        try {
            sensorDataList = data;
            sensorDataList.add((float) accuracy);

            ursAI2Sensor.SensorChanged(sensorType.toUnderlyingValue(), YailList.makeList(sensorDataList),
                    accuracy);
        } catch (Exception e) {
            DebugUtil.LogExecption(e);
        }
    }
}
