package de.ullisroboterseite.ursai2sensor;

import com.google.appinventor.components.common.Sensitivity;
import com.google.appinventor.components.runtime.util.SdkLevel;
import com.google.appinventor.components.runtime.util.YailList;

import android.content.res.Configuration;
import android.content.res.Resources;
import android.os.Build;
import android.os.Handler;
import android.content.Context;
import android.hardware.Sensor;
import android.hardware.SensorEvent;

import android.util.Log;
import android.view.Surface;
import android.view.WindowManager;

import android.hardware.SensorEventListener;

import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import java.util.Queue;
import java.util.ResourceBundle.Control;

public class ShakingSensor extends SensorBase {

    Sensitivity sensitivity;

    // Shake thresholds - derived by trial
    static final double weakShakeThreshold = 5.0;
    static final double moderateShakeThreshold = 13.0;
    static final double strongShakeThreshold = 20.0;

    // Cache for shake detection
    static final int SENSOR_CACHE_SIZE = 10;
    final Queue<Float> X_CACHE = new LinkedList<Float>();
    final Queue<Float> Y_CACHE = new LinkedList<Float>();
    final Queue<Float> Z_CACHE = new LinkedList<Float>();

    //Specifies the minimum time interval between calls to Shaking()
    int minimumInterval = 400;

    //Specifies the time when Shaking() was last called
    long timeLastShook;

    public ShakingSensor(UrsAI2Sensor ursAI2Sensor) {
        super(ursAI2Sensor, SensorType.ShakingSensor, Sensor.TYPE_ACCELEROMETER);
    }

    // The minimum delay allowed between two events in microseconds
    @Override
    public int getMinDelay() {
        return androidSensor.getMinDelay() * (SENSOR_CACHE_SIZE + 5); // + 5 for safety
    }

    // The maximum delay allowed between two events in microseconds
    @Override
    public int getMaxDelay() {
        return 1000000; // 1 sec
    }

    @Override
    public void onStart(int argDelay, YailList control) {
        minimumInterval = delay / 1000; // us -> ms
        delay = argDelay / (SENSOR_CACHE_SIZE + 5); // + 5 for safety to get enough readings in 'minimumInterval'
        try {
            Log.d(LOG_TAG, "getString: " + control.getString(0));
            sensitivity = Sensitivity.valueOf(control.getString(0));
        } catch (Exception e) {
            sensitivity = Sensitivity.Weak;
            ursAI2Sensor.thisForm.ErrorOccurred(ursAI2Sensor, "ShakingSensor", 16093,
                    "Bad value for Sensivity. Replaced with 'Weak'.");

        }

        X_CACHE.clear();
        Y_CACHE.clear();
        Z_CACHE.clear();
        timeLastShook = 0;
        Log.d(LOG_TAG, "onStart ShakingSensor, minimumInterval: " + minimumInterval
                + " delay: " + delay + " Sensivity: " + sensitivity.toString());
    }

    @Override
    public void onStop() {
        // nothing to do
    }

    // Updating sensor cache, replacing oldest values.
    void addToSensorCache(Queue<Float> cache, float value) {
        if (cache.size() >= SENSOR_CACHE_SIZE) {
            cache.remove();
        }
        cache.add(value);
    }

    // SensorListener implementation
    @Override
    public void onSensorChanged(SensorEvent sensorEvent) {
        addToSensorCache(X_CACHE, sensorEvent.values[0]);
        addToSensorCache(Y_CACHE, sensorEvent.values[1]);
        addToSensorCache(Z_CACHE, sensorEvent.values[2]);

        long currentTime = System.currentTimeMillis();

        // Checks whether the phone is shaking and the minimum interval
        // has elapsed since the last registered a shaking event.
        if ((isShaking(X_CACHE, sensorEvent.values[0]) || isShaking(Y_CACHE, sensorEvent.values[1])
                || isShaking(Z_CACHE, sensorEvent.values[2]))
                && (timeLastShook == 0 || currentTime >= timeLastShook + minimumInterval)) {
            timeLastShook = currentTime;

            reportSensorData(sensorType, new ArrayList<Float>(), Accuracy.Meaningless.toUnderlyingValue());
        }
    }

    // Indicates whether there was a sudden, unusual movement.
    // See http://www.utdallas.edu/~rxb023100/pubs/Accelerometer_WBSN.pdf.
    boolean isShaking(Queue<Float> cache, float currentValue) {
        float average = 0;
        for (float value : cache) {
            average += value;
        }
        average /= cache.size();
        float delta = Math.abs(average - currentValue);

        switch (sensitivity) {
            case Weak:
                return delta > strongShakeThreshold;
            case Moderate:
                return delta > moderateShakeThreshold && delta < strongShakeThreshold;
            case Strong:
                return delta > weakShakeThreshold && delta < moderateShakeThreshold;
            default:
                return false;
        }
    }
}
