package de.ullisroboterseite.ursai2sensorutil;

// Autor: http://UllisRoboterSeite.de

// Doku:  http://UllisRoboterSeite.de/android-AI2-Sensorutil.html
//
// Version 1.0 (2025-10-08)
// -------------------------
// - Basis-Version
//

import com.google.appinventor.components.annotations.*;
import com.google.appinventor.components.annotations.androidmanifest.*;
import com.google.appinventor.components.common.*;
import com.google.appinventor.components.runtime.*;
import com.google.appinventor.components.runtime.util.*;

import android.hardware.Sensor;

import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.content.BroadcastReceiver;
import android.content.ComponentName;
import android.content.ServiceConnection;

import android.os.Build;
import android.os.Handler;
import android.os.Binder;
import android.os.IBinder;
import android.os.PowerManager;
import android.os.PowerManager.WakeLock;

import android.net.wifi.WifiManager;
import android.net.Uri;

import android.app.Notification;
import android.app.ActivityManager;
import android.provider.Settings;

import androidx.core.app.NotificationManagerCompat;

import android.util.Log;

import java.util.*;

import android.Manifest;
import android.content.pm.PackageManager;

@DesignerComponent(version = 1, //
        versionName = UrsAI2SensorUtil.VersionName, //
        dateBuilt = UrsAI2SensorUtil.dateBuilt, //
        description = "AI2 extension block for sensor support.", //
        category = com.google.appinventor.components.common.ComponentCategory.EXTENSION, //
        nonVisible = true, //
        helpUrl = "http://UllisRoboterSeite.de/android-AI2-Sensorutil.html", //
        iconName = "aiwebres/icon.png")

@SimpleObject(external = true)
@UsesPermissions(permissionNames = "android.permission.FOREGROUND_SERVICE, android.permission.FOREGROUND_SERVICE_MEDIA_PLAYBACK"
        + ", android.permission.WAKE_LOCK, android.permission.POST_NOTIFICATIONS, android.permission.REQUEST_IGNORE_BATTERY_OPTIMIZATIONS")
@UsesServices(services = {
        @ServiceElement(name = "de.ullisroboterseite.ursai2sensorutil.SensorService", enabled = "true", exported = "true", foregroundServiceType = "mediaPlayback") })

public class UrsAI2SensorUtil extends AndroidNonvisibleComponent
        implements OnPauseListener, OnResumeListener, OnStopListener, Deleteable,
        OnInitializeListener, OnClearListener, OnDestroyListener {

    final static public String LOG_TAG = "Sensor";
    static final String VersionName = "1.0.-1";
    static final String dateBuilt = "2025-10-11";

    final Handler androidUIHandler; // Used to launch Runnables on the UI Thread after a delay

    static public UrsAI2SensorUtil thisInstance; // ToDo: Das ist nicht gut!
    public Form thisForm; // ToDo: Das ist nicht gut!

    public boolean isInitialized = false; // Is set to true at end of onIntialize.
    public boolean isFGSstarted = false; // Indicates wether the FGS has been started.

    public boolean checkForIsIgnoringBatteryOptimization = false;
    // Nach RequestIgnoreBatteryOptimizations wird  die Kombination onPause-OnResume ausglöst.
    // Beim onResume wurde der Dialog zu RequestIgnoreBatteryOptimizations geschlossen.

    public String packageName; // Package name for this app.

    ServiceConnection connection;
    SensorService sensorService = null; // Reference to SensorService instance
    NotificationManagerCompat notificationManager; // https://developer.android.com/reference/androidx/core/app/NotificationManagerCompat
    PowerManager powerManager; // https://developer.android.com/reference/android/os/PowerManager
    PowerManager.WakeLock wakeLock = null; // https://developer.android.com/reference/android/os/PowerManager.WakeLock
    WifiManager.WifiLock wifiLock = null; // Verweis auf das gesetzte WiFiLock. null, wenn kein Lock gesetzt.

    // ----- Properties -----
    boolean aquireWifiLock = false;

    // ----- Channel -----
    static final String channelID = "UrsSensorChannel";
    String channelName = "URS Sensor";
    String channelDescription = "";
    int channelImportance = NotificationManagerCompat.IMPORTANCE_DEFAULT;

    // ----- Notification -----
    String notificationIcon = "ic_launcher";
    String notificationTitle = "Sensor Service";
    String notificationText = "";
    String screenToOpen = "";
    String startValue = "SENSOR";
    String smallIconPath = "";
    String largeIconPath = "";
    boolean notificationDeletable = false;
    String action1Title = "";
    String action2Title = "";
    String action3Title = "";

    int notificationIDBase = 16444;
    int notificationID; // The ID of the notification. Adjusted in the constructor!

    static final String DELETE_TAG = "URS_NOTIFICATION_DELETE";
    static final String NOTIFICATION_ID_TAG = "NOTIFICATION_ID";
    static final String ACTION_ID_TAG = "ACTION_ID";
    static final String NOTIFICATION_EVENT_TAG = "URS_NOTIFICATION_EVENT";
    static final int REQUEST_NOTIFICATION_PERMISSION = 1;

    // =====================================================
    // === BroadcastReceivers                       ========
    // =====================================================
    BroadcastReceiver onClickReceiver = new BroadcastReceiver() {
        @Override
        public synchronized void onReceive(final Context arg0, final Intent arg1) {
            int notificationId = arg1.getIntExtra(NOTIFICATION_ID_TAG, -9999);
            int actionID = arg1.getIntExtra(ACTION_ID_TAG, -9);
            if (notificationId == notificationID) {
                OnActionClick(actionID);
            }
        }
    };

    BroadcastReceiver deleteReceiver = new BroadcastReceiver() {
        @Override
        public synchronized void onReceive(final Context arg0, final Intent arg1) {
            Log.d(LOG_TAG, "deleteReceiver");
            int id = arg1.getIntExtra(NOTIFICATION_ID_TAG, -9999);
            if (id == notificationID && !notificationDeletable)
                UpdateNotification();
            else
                OnDeleteNotification();
        }
    };

    // =====================================================
    // === Code                                     ========
    // =====================================================
    public UrsAI2SensorUtil(ComponentContainer container) {
        super(container.$form());

        Log.d(LOG_TAG, "ctor: " + this.getClass().getSimpleName() + " - " + VersionName + " IsRunningInCompanion: "
                + IsRunningInCompanion());

        thisInstance = this;
        thisForm = container.$form();
        notificationIDBase++;
        notificationID = notificationIDBase;

        androidUIHandler = new Handler();

        form.registerForOnInitialize(this);
        form.registerForOnPause(this);
        form.registerForOnResume(this);
        form.registerForOnStop(this);
        form.registerForOnClear(this);
        form.registerForOnDestroy(this);

        connection = new ServiceConnection() {
            @Override
            public void onServiceConnected(ComponentName name, IBinder service) {
                SensorService.LocalBinder binder = (SensorService.LocalBinder) service;
                sensorService = binder.getService();
                Log.d(LOG_TAG, "onServiceConnected");
            }

            @Override
            public void onServiceDisconnected(ComponentName name) {
                // Handle service disconnection
                sensorService = null;
                Log.d(LOG_TAG, "onServiceDisconnected");
            }
        };

        powerManager = (PowerManager) form.getSystemService(Context.POWER_SERVICE);
        notificationManager = NotificationManagerCompat.from(form);

        form.registerReceiver(onClickReceiver, new IntentFilter(NOTIFICATION_EVENT_TAG), Context.RECEIVER_EXPORTED);
        form.registerReceiver(deleteReceiver, new IntentFilter(DELETE_TAG), Context.RECEIVER_EXPORTED);

        packageName = form.getPackageName();
    } // ctor

    // onInitializeListener implementation
    @Override
    public void onInitialize() {
        Log.d(LOG_TAG, "onInitialize");
        if (!IsRunningInCompanion())
            Util.createChannel(form, channelID, channelName, channelDescription, channelImportance);

        isInitialized = true;
    }

    // =====================================================
    // === Properties                               ========
    // =====================================================

    // Returns the version of this Extension.
    @SimpleProperty(description = "Returns the component's version name.")
    public String Version() {
        return VersionName;
    }

    // Return the current Android-SDK-Version.
    @SimpleProperty(description = "Returns the running Android SDK version.")
    public int VersionSDK() {
        return Build.VERSION.SDK_INT;
    }

    // Returns wether the App can post notifications.
    // https://developer.android.com/reference/android/app/NotificationManager#areNotificationsEnabled()
    // ab API Level 24
    @SimpleProperty(description = "Returns whether notifications are enabled for this app.")
    public boolean AreNotificationsEnabled() {
        if (Build.VERSION.SDK_INT >= 24) {
            return notificationManager.areNotificationsEnabled();
        } else
            return true;
    }

    @SimpleProperty(description = "Returns wether the app is running in the companion.")
    public boolean IsRunningInCompanion() {
        return form instanceof ReplForm;
    }

    @SimpleProperty(description = "Returns wether the app is running in the companion.")
    public boolean IsIgnoringBatteryOptimizations() {
        if (Build.VERSION.SDK_INT >= 23) {
            return powerManager.isIgnoringBatteryOptimizations(form.getPackageName());
        }
        return true; // else
    }

    @DesignerProperty(editorType = PropertyTypeConstants.PROPERTY_TYPE_BOOLEAN, defaultValue = "False")
    @SimpleProperty(description = "Specifies wether a WiFilock should be aquired when starting the foregroundservice.")
    public void AquireWiFiLock(boolean value) {
        aquireWifiLock = value;
    }

    @SimpleProperty(category = PropertyCategory.BEHAVIOR, description = "Specifies wether a WiFilock should be aquired when starting the foregroundservice.")
    public boolean AquireWiFiLock() {
        return aquireWifiLock;
    }

    // =====================================================
    // === Channel Properties                       ========
    // =====================================================

    // https://developer.android.com/reference/android/app/NotificationChannel#setName(java.lang.CharSequence)
    @DesignerProperty(editorType = PropertyTypeConstants.PROPERTY_TYPE_TEXT, alwaysSend = true, defaultValue = "URS Sensor")
    @SimpleProperty(description = "The the user visible name of this channel.")
    public void ChannelName(String value) {
        value = value.trim();
        if (value.isEmpty())
            value = "URS Sensor";
        channelName = value;
        if (!IsRunningInCompanion())
            Util.createChannel(form, channelID, channelName, channelDescription, channelImportance);
    }

    @SimpleProperty(category = PropertyCategory.APPEARANCE, description = "The the user visible name of this channel.")
    public String ChannelName() {
        return channelName;
    }

    // https://developer.android.com/reference/android/app/NotificationChannel#setImportance(int)
    // nicht änderbar
    @DesignerProperty(editorType = PropertyTypeConstants.PROPERTY_TYPE_CHOICES, defaultValue = "Default", editorArgs = {
            "None", "Min", "Low", "Default", "High", "Max" })
    @SimpleProperty(category = PropertyCategory.BEHAVIOR, description = "The level of interruption of this notification channel.", userVisible = false)
    public void ChannelImportance(String value) {
        if (value.equals("None"))
            channelImportance = NotificationManagerCompat.IMPORTANCE_NONE;
        else if (value.equals("Min"))
            channelImportance = NotificationManagerCompat.IMPORTANCE_MIN;
        else if (value.equals("Low"))
            channelImportance = NotificationManagerCompat.IMPORTANCE_LOW;
        else if (value.equals("Default"))
            channelImportance = NotificationManagerCompat.IMPORTANCE_DEFAULT;
        else if (value.equals("High"))
            channelImportance = NotificationManagerCompat.IMPORTANCE_HIGH;
        else if (value.equals("Max"))
            channelImportance = NotificationManagerCompat.IMPORTANCE_MAX;
        else
            channelImportance = NotificationManagerCompat.IMPORTANCE_UNSPECIFIED;

        Log.d(LOG_TAG, "Channel importance is " + value + " = " + channelImportance);
    }

    @DesignerProperty(editorType = PropertyTypeConstants.PROPERTY_TYPE_TEXT, defaultValue = "")
    @SimpleProperty(category = PropertyCategory.APPEARANCE, description = "The user visible description of this channel.")
    public void ChannelDescription(String value) {
        channelDescription = value.trim();
        if (!IsRunningInCompanion())
            Util.createChannel(form, channelID, channelName, channelDescription, channelImportance);
    }

    @SimpleProperty(category = PropertyCategory.APPEARANCE, description = "The internal id of this channel.")
    public String ChannelID() {
        return channelDescription;
    }

    // =====================================================
    // === Notification Properties                  ========
    // =====================================================

    @SimpleProperty(category = PropertyCategory.APPEARANCE, description = "Gets the ID of the notification.")
    public int NotificationID() {
        return notificationID;
    }

    @DesignerProperty(editorType = PropertyTypeConstants.PROPERTY_TYPE_TEXT, defaultValue = "Sensor Service")
    @SimpleProperty(description = "Specifies the title of a desired notification.")
    public void NotificationTitle(String value) {
        notificationTitle = value;
        UpdateNotification();
    }

    @SimpleProperty(category = PropertyCategory.APPEARANCE, description = "Specifies the title of the notification.")
    public String NotificationTitle() {
        return notificationTitle;
    }

    @DesignerProperty(editorType = PropertyTypeConstants.PROPERTY_TYPE_TEXT, defaultValue = "")
    @SimpleProperty(description = "Specifies the text of a desired notification.")
    public void NotificationText(String value) {
        notificationText = value;
        UpdateNotification();
    }

    @SimpleProperty(category = PropertyCategory.APPEARANCE, description = "Specifies the text of the notification.")
    public String NotificationText() {
        return notificationText;

    }

    @DesignerProperty(editorType = PropertyTypeConstants.PROPERTY_TYPE_TEXT, defaultValue = "ic_launcher")
    @SimpleProperty(category = PropertyCategory.APPEARANCE, description = "The resource name of a drawable to use as the icon in the status bar.")
    public void NotificationIcon(String value) {
        notificationIcon = value.trim();
        UpdateNotification();
    }

    @SimpleProperty(category = PropertyCategory.APPEARANCE, description = "The resource name of a drawable to use as the icon in the status bar.")
    public String NotificationIcon() {
        return notificationIcon;
    }

    // https://developer.android.com/reference/android/app/Notification.Builder#setSmallIcon(android.graphics.drawable.Icon)
    @DesignerProperty(editorType = PropertyTypeConstants.PROPERTY_TYPE_ASSET, defaultValue = "")
    @SimpleProperty(description = "The small icon representing this notification in the status bar and content view.")
    public void NotificationIconAsset(String value) {
        smallIconPath = value != null ? value.trim() : "";
        UpdateNotification();
    }

    @SimpleProperty(category = PropertyCategory.APPEARANCE, description = "The small icon representing this notification in the status bar and content view.")
    public String NotificationIconAsset() {
        return smallIconPath;
    }

    // https://developer.android.com/reference/android/app/Notification.Builder#setSmallIcon(android.graphics.drawable.Icon)
    @DesignerProperty(editorType = PropertyTypeConstants.PROPERTY_TYPE_ASSET, defaultValue = "")
    @SimpleProperty(description = "The large icon for the notification.")
    public void NotificationLargeIcon(String value) {
        largeIconPath = value != null ? value.trim() : "";
        UpdateNotification();
    }

    @SimpleProperty(category = PropertyCategory.APPEARANCE, description = "The large icon for the notification.")
    public String NotificationLargeIcon() {
        return largeIconPath;
    }

    @DesignerProperty(editorType = PropertyTypeConstants.PROPERTY_TYPE_BOOLEAN, defaultValue = "False")
    @SimpleProperty(category = PropertyCategory.BEHAVIOR, description = "The notification can be dismissed by the user.")
    public void NotificationDeletable(boolean value) {
        notificationDeletable = value;
        UpdateNotification();
    }

    @SimpleProperty(category = PropertyCategory.BEHAVIOR, description = "The notification can be dismissed by the user.")
    public boolean NotificationDeletable() {
        return notificationDeletable;
    }

    @DesignerProperty(editorType = PropertyTypeConstants.PROPERTY_TYPE_TEXT, defaultValue = "")
    @SimpleProperty(category = PropertyCategory.BEHAVIOR, description = "The Screen that will be opened when the notifaction is clicked.")
    public void ScreenToOpen(String value) {
        screenToOpen = value.trim();
        UpdateNotification();
    }

    @SimpleProperty(category = PropertyCategory.BEHAVIOR, description = "The Screen that will be opened when the notifaction is clicked.")
    public String ScreenToOpen() {
        return screenToOpen;
    }

    @DesignerProperty(editorType = PropertyTypeConstants.PROPERTY_TYPE_TEXT, defaultValue = "SENSOR")
    @SimpleProperty(category = PropertyCategory.BEHAVIOR, description = "The start value for a new opened screen.")
    public void StartValue(String value) {
        startValue = value.trim();
        UpdateNotification();
    }

    @SimpleProperty(category = PropertyCategory.BEHAVIOR, description = "The start value for a new opened screen.")
    public String StartValue() {
        return startValue;
    }

    @DesignerProperty(editorType = PropertyTypeConstants.PROPERTY_TYPE_TEXT, defaultValue = "")
    @SimpleProperty(category = PropertyCategory.APPEARANCE, description = "The title for action button #1.")
    public void Action1Title(String value) {
        action1Title = value.trim();
        UpdateNotification();
    }

    @SimpleProperty(category = PropertyCategory.APPEARANCE, description = "The title for action button #1.")
    public String Action1Title() {
        return action1Title;
    }

    @DesignerProperty(editorType = PropertyTypeConstants.PROPERTY_TYPE_TEXT, defaultValue = "")
    @SimpleProperty(category = PropertyCategory.APPEARANCE, description = "The title for action button #2.")
    public void Action2Title(String value) {
        action2Title = value.trim();
        UpdateNotification();
    }

    @SimpleProperty(category = PropertyCategory.APPEARANCE, description = "The title for action button #2.")
    public String Action2Title() {
        return action2Title;
    }

    @DesignerProperty(editorType = PropertyTypeConstants.PROPERTY_TYPE_TEXT, defaultValue = "")
    @SimpleProperty(category = PropertyCategory.APPEARANCE, description = "The title for action button #3.")
    public void Action3Title(String value) {
        action3Title = value.trim();
        UpdateNotification();
    }

    @SimpleProperty(category = PropertyCategory.APPEARANCE, description = "The title for action button #3.")
    public String Action3Title() {
        return action3Title;
    }

    // =====================================================
    // === Functions                                ========
    // =====================================================
    @SimpleFunction(description = "Ask the user to allow the app to ignore battery optimizations.")
    public void RequestIgnoreBatteryOptimizations() {
        if (Build.VERSION.SDK_INT >= 23) {
            try {
                checkForIsIgnoringBatteryOptimization = true;
                Intent intent = new Intent(android.provider.Settings.ACTION_REQUEST_IGNORE_BATTERY_OPTIMIZATIONS);
                intent.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
                intent.setData(Uri.parse("package:" + form.getPackageName()));
                form.startActivity(intent);
            } catch (Exception e) {
                DebugUtil.LogExecption(e);
            }
        }
    }

    @SimpleFunction(description = "Starts the foregroundservice")
    public void StartForegroundService() {
        Log.d(LOG_TAG, "startForegroundService: " + this.getClass().getSimpleName() + " - " + VersionName);

        String className = getClassForScreen(screenToOpen);
        try {
            Intent serviceIntent = new Intent(form, SensorService.class);
            serviceIntent.putExtra("NotificationID", notificationID);
            serviceIntent.putExtra("Icon", Util.getSystemIcon(notificationIcon, form));
            serviceIntent.putExtra("SmallIconPath", smallIconPath);
            serviceIntent.putExtra("LargeIconPath", largeIconPath);
            serviceIntent.putExtra("Title", notificationTitle);
            serviceIntent.putExtra("Text", notificationText);
            serviceIntent.putExtra("StartValue", startValue);
            serviceIntent.putExtra("Deletable", notificationDeletable);
            serviceIntent.putExtra("ClassName", className);
            serviceIntent.putExtra("ChannelID", channelID);
            serviceIntent.putExtra("Action1", action1Title);
            serviceIntent.putExtra("Action2", action2Title);
            serviceIntent.putExtra("Action3", action3Title);

            thisInstance = this; // Needed in FGS
            Object cmp = form.startForegroundService(serviceIntent);

            if (cmp == null) {
                Log.d(LOG_TAG, "FGS gestartet");
                form.ErrorOccurred(this, "SensorService", 16093, "Cannot start ForeGroundService.");
                return;
            }

            // Bind to the sensor service
            form.bindService(serviceIntent, connection, Context.BIND_AUTO_CREATE);
            isFGSstarted = true;

            // Request a WakeLock
            wakeLock = powerManager.newWakeLock(PowerManager.PARTIAL_WAKE_LOCK, "UrsSensor-WakeLock");
            wakeLock.acquire();
            Log.d(LOG_TAG, "WakeLock acquired");

            if (aquireWifiLock) {
                WifiManager wm = (WifiManager) form.getSystemService(Context.WIFI_SERVICE);
                wifiLock = wm.createWifiLock(WifiManager.WIFI_MODE_FULL_HIGH_PERF, "UrsSensor-UrsWifiLock");
                Log.d(LOG_TAG, "WifiLock acquired");
            }

            AfterServiceStarted();
        } catch (Exception e) {
            DebugUtil.LogExecption(e);
        }
    }

    @SimpleFunction(description = "Stops the foregroundservice")
    public void StopForegroundService() {
        if (wakeLock != null) {// Currently no WakeLock.
            wakeLock.release();
            wakeLock = null;
        }
        if (wifiLock != null) {
            wifiLock.release();
            wifiLock = null;
        }

        if (isFGSstarted) {
            try {
                Log.d(LOG_TAG, "stopService: notficationID: " + notificationID);
                form.unbindService(connection);
                form.stopService(new Intent(form, SensorService.class));
            } catch (Exception e) {
                DebugUtil.LogExecption("StopForegroundService: ", e);
                form.ErrorOccurred(this, "SensorService", 16092, "StopForegroundService: " + e.toString());
            }
        }
    }

    @SimpleFunction(description = "Requests the permission to post notifications.")
    public void RequestNotificationPermission() {
        if (Build.VERSION.SDK_INT < 33) // POST_NOTIFICATIONS was introduced in version 33.
            OnNotificationPermissionResponse(true);
        else {
            if (form.checkSelfPermission(Manifest.permission.POST_NOTIFICATIONS) == PackageManager.PERMISSION_GRANTED) {
                OnNotificationPermissionResponse(true);
            } else {
                form.runOnUiThread(new Runnable() {
                    @Override
                    public void run() {
                        form.askPermission(Manifest.permission.POST_NOTIFICATIONS, new PermissionResultHandler() {
                            @Override
                            public void HandlePermissionResponse(String permission, boolean granted) {
                                OnNotificationPermissionResponse(granted);
                            }
                        });
                    }
                });
            }
        }
    }

    // OnPauseListener implementation
    @Override
    public void onPause() {
        Log.d(LOG_TAG, "onPause");
        OnPause(); // Raise event
    }

    // OnResumeListener implementation
    @Override
    public void onResume() {
        Log.d(LOG_TAG, "onResume");
        OnResume(); // Raise event

        // Die Kombination onPause-OnResume wird nach RequestIgnoreBatteryOptimizations ausgelöst

        if (checkForIsIgnoringBatteryOptimization ) {
            checkForIsIgnoringBatteryOptimization = false;
            OnBackgroundPermissionResponse(IsIgnoringBatteryOptimizations());
        }
    }

    // OnResumeListener implementation
    @Override
    public void onStop() {
        Log.d(LOG_TAG, "onStop");
        OnStop(); // Raise event
    }

    // Deleteable implementation
    @Override
    public void onDelete() {
        Log.d(LOG_TAG, "onDelete");
        StopForegroundService();
    }

    @Override
    public void onClear() {
        Log.d(LOG_TAG, "onClear");
        StopForegroundService();
    }

    @Override
    public void onDestroy() {
        Log.d(LOG_TAG, "onDestroy");
        StopForegroundService();
    }

    String getClassForScreen(String screenToOpen) {
        String className = "";
        if (!screenToOpen.isEmpty()) { // Leaving this field blank will not result in any subsequent action.
            className = form.getClass().getPackage().getName() + "." + screenToOpen; // Screen of this AI2-App.
            try { // Check whether the class is suitable as an intent.
                Intent notificationIntent = new Intent(form, Class.forName(className));
                Log.d(LOG_TAG, "SensorService, Klasse ok: " + className);
            } catch (Exception e) {
                Log.d(LOG_TAG, "SensorService: Klasse nicht gefunden: " + className);
                form.ErrorOccurred(this, "SensorService", 16090, "Screen not found: " + screenToOpen);
                className = "";
            }
        }
        return className;
    }

    public void UpdateNotification() {
        if (!isInitialized)
            return;
        if (!isFGSstarted)
            return;

        Log.d(LOG_TAG, "Update notification");
        String className = getClassForScreen(screenToOpen);
        Notification notification = Util.CreateNotification(form, channelID, notificationTitle, notificationText,
                smallIconPath, Util.getSystemIcon(notificationIcon, form), className, startValue, largeIconPath,
                notificationID, notificationDeletable, action1Title, action2Title, action3Title);

        Log.d(LOG_TAG, "Update notification, ID: " + notificationID + " notification: " + notification.toString());

        notificationManager.notify(notificationID, notification);
    }

    // =====================================================
    // === Events                                   ========
    // =====================================================
    @SimpleEvent(description = "The app is paused.")
    public void OnPause() {
        EventDispatcher.dispatchEvent(this, "OnPause");
    }

    @SimpleEvent(description = "The app is resumed.")
    public void OnResume() {
        EventDispatcher.dispatchEvent(this, "OnResume");
    }

    @SimpleEvent(description = "Triggered when app is stopped.")
    public void OnStop() {
        EventDispatcher.dispatchEvent(this, "OnStop");
    }

    @SimpleEvent(description = "The notification or an action button was clicked by the user.")
    public void OnActionClick(final int ActionNo) {
        androidUIHandler.post(new Runnable() { // Ggf. wird dieses Ereignis in einem Thread ausgelöst.
            public void run() {
                EventDispatcher.dispatchEvent(thisInstance, "OnActionClick", ActionNo);
            }
        });
    }

    @SimpleEvent(description = "Indicates the change of sensor data.")
    public void OnDeleteNotification() {
        EventDispatcher.dispatchEvent(this, "OnDeleteNotification");
    }

    @SimpleEvent(description = "Indicates that the foreground service was started.")
    public void AfterServiceStarted() {
        EventDispatcher.dispatchEvent(this, "AfterServiceStarted");
    }

    @SimpleEvent(description = "Returns the result of RequestNotificationPermission.")
    public void OnNotificationPermissionResponse(boolean Granted) {
        EventDispatcher.dispatchEvent(this, "OnNotificationPermissionResponse", Granted);
    }

    @SimpleEvent(description = "Returns the result of RequestIgnoreBatteryOptimizations.")
    public void OnBackgroundPermissionResponse(boolean Granted) {
        EventDispatcher.dispatchEvent(this, "OnBackgroundPermissionResponse", Granted);
    }

    @SimpleFunction(description = "Removes onPause and onResume listeners.")
    public void RemoveListeners(com.google.appinventor.components.runtime.Component Sensor) {
        try {
            Set<OnPauseListener> onPauseListeners = (Set<OnPauseListener>) DebugUtil.getPrivateField(form,
                    "onPauseListeners");
            onPauseListeners.remove(Sensor);
            Set<OnResumeListener> onResumeListeners = (Set<OnResumeListener>) DebugUtil.getPrivateField(form,
                    "onResumeListeners");
            onResumeListeners.remove(Sensor);
            Log.d(LOG_TAG, "Listeners removed from " + Sensor.getClass().toString());
        } catch (Exception e) {
            DebugUtil.LogExecption(e);
        }
    }
}