package de.ullisroboterseite.ursai2sensorutil;

import java.util.Arrays;
import java.util.LinkedList;
import java.util.Queue;

import android.util.*;

import com.google.appinventor.components.annotations.*;
import com.google.appinventor.components.annotations.androidmanifest.*;
import com.google.appinventor.components.common.*;
import com.google.appinventor.components.runtime.*;
import com.google.appinventor.components.runtime.util.*;

@DesignerComponent(version = 1, //
        versionName = UrsAI2SensorUtil.VersionName, //
        dateBuilt = UrsAI2SensorUtil.dateBuilt, //
        description = "AI2 extension block for sensor support.", //
        category = com.google.appinventor.components.common.ComponentCategory.EXTENSION, //
        nonVisible = true, //
        helpUrl = "http://UllisRoboterSeite.de/android-AI2-Sensorutil.html", //
        iconName = "aiwebres/savg.png")
@SimpleObject(external = true)

public class UrsSensorAverager extends AndroidNonvisibleComponent
        implements DataSourceChangeListener {
    static final String LOG_TAG = UrsAI2SensorUtil.LOG_TAG;

    // Represents the key value of the value to use from the attached Data Source.
    protected String dataSourceKey;

    // Property used in Designer to import from a Data Source.
    private DataSource<?, ?> dataSource; // Attached Chart Data Source

    // Queue to hold the data to be averaged
    final LinkedList<Float> dataCache = new LinkedList<Float>();

    int cacheSize = 5;

    public UrsSensorAverager(ComponentContainer container) {
        super(container.$form());
    }

    /**
    * Sets the Data Source key identifier for the value to import from the
    * attached Data Source.
    *
    *   An example is the tag of the TinyDB component, which identifies the value.
    *
    *   The property is a Designer-only property, and should be changed after setting the
    * Source component of the Chart Data component.
    *
    *   A complete list of applicable values for each compatible source is as follows:
    *
    *     * For the AccelerometerSensor, the value should be one of the following: X Y or Z
    *     * For the GyroscopeSensor, the value should be one of the following: X Y or Z
    *     * For the LocationSensor, the value should be one of the following:
    *       latitude, longitude, altitude or speed
    *     * For the OrientationSensor, the value should be one of the following:
    *       pitch, azimuth or roll
    *     * For the Pedometer, the value should be one of the following:
    *       WalkSteps, SimpleSteps or Distance
    *     * For the ProximitySensor, the value should be distance.
    *       For instance, if values come in the format "t:12", the prefix can be specified as "t:",
    *       and the prefix will then be removed from the data. No value can be specified if purely
    *       numerical values are returned.
    *
    * @param key new key value
    */
    @DesignerProperty(editorType = PropertyTypeConstants.PROPERTY_TYPE_STRING)
    @SimpleProperty(category = PropertyCategory.BEHAVIOR, userVisible = false, //
            description = "Sets the Data Source key identifier for the value to import from the attached Data Source. ")
    public void DataSourceKey(String key) {
        this.dataSourceKey = key;
    }

    /**
      * Sets the Source to use for the Data component. Valid choices
      * include AccelerometerSensor, BluetoothClient, CloudDB, DataFile,
      * GyroscopeSensor, LocationSesnro, OrientationSensor, Pedometer,
      * ProximitySensor TinyDB and Web components. The Source value also requires
      * valid DataSourceValue, WebColumn or DataFileColumn properties,
      * depending on the type of the Source attached (the required properties
      * show up in the Properties menu after the Source is changed).
      *
      *   If the data identified by the {@link #DataSourceKey(String)} is updated
      * in the attached Data Source component, then the data is also updated in
      * the Chart Data component.
      *
      * @param dataSource Data Source to use for the Chart data.
      */
    @SimpleProperty(category = PropertyCategory.BEHAVIOR, userVisible = false, description = "Sets the Source to use for the Data component. "
            + "Valid choices include AccelerometerSensor, GyroscopeSensor, LocationSensor, OrientationSensor, Pedometer, ProximitySensor.")
    @DesignerProperty(editorType = PropertyTypeConstants.PROPERTY_TYPE_CHART_DATA_SOURCE)
    public <K, V> void Source(DataSource<K, V> dataSource) {
        if (!(dataSource instanceof SensorComponent)) // Only Sensors
            return;

        // If the previous Data Source is an ObservableDataSource,
        // this Chart Data component must be removed from the observers
        // List of the Data Source.
        if (this.dataSource != dataSource && this.dataSource instanceof ObservableDataSource) {
            ((ObservableDataSource<?, ?>) this.dataSource).removeDataObserver(this);
        }

        this.dataSource = dataSource;

        if (dataSource instanceof ObservableDataSource) {
            // Add this Data Component as an observer to the ObservableDataSource object
            ((ObservableDataSource<?, ?>) dataSource).addDataObserver(this);
        }
        Log.d(LOG_TAG, "DataSource added: " + dataSource.getClass().toString());
    }

    @DesignerProperty(editorType = PropertyTypeConstants.PROPERTY_TYPE_NON_NEGATIVE_INTEGER, defaultValue = "5")
    @SimpleProperty(category = PropertyCategory.BEHAVIOR, description = "Specifies the number of values to aggregate.")
    public void CacheSize(int value) {
        this.cacheSize = value;
    }

    @SimpleProperty(category = PropertyCategory.BEHAVIOR, description = "Specifies the number of values to aggregate.")
    public int CacheSize() {
        return cacheSize;
    }

    @SimpleProperty(category = PropertyCategory.BEHAVIOR, description = "Gets the number of stored values.")
    public int Count() {
        return dataCache.size();
    }

    @SimpleProperty(category = PropertyCategory.BEHAVIOR, description = "Calcutes the average of the stored values.")
    public float Average() {
        float average = 0;
        for (float value : dataCache) {
            average += value;
        }
        average /= dataCache.size();

        return average;
    }

    @SimpleProperty(category = PropertyCategory.BEHAVIOR, description = "Calcutes the weighted average of the stored values.")
    public float WeightedAverage() {
        float average = 0;
        int divider = 0;

        for (int i = 0; i < dataCache.size(); i++) {
            average += dataCache.get(i) * i;
            divider += i;
        }
        average /= divider;

        return average;
    }

    /**
     * Event called when a new real time value is sent to the observer.
     *
     * @param component  component that triggered the event
     * @param key  identifier of the value
     * @param value  value received
     */
    @Override
    public void onReceiveValue(RealTimeDataSource<?, ?> component, final String key, Object value) {
        // Calling component is not the actual Data Source
        if (component != dataSource) {
            return;
        }

        if (key.equals(dataSourceKey)) {
            if (dataCache.size() >= cacheSize) {
                dataCache.remove();
            }
            dataCache.add((float) value);
        }
    }

    /**
    * Event called when the value of the observed DataSource component changes.
    *
    * @param component component that triggered the event
    * @param key       key of the value that changed
    * @param newValue  the new value of the observed value
    */
    @Override
    public void onDataSourceValueChange(DataSource<?, ?> component, String key, Object newValue) {
    }
}