#include "M28BYJ_48.h"

M28BYJ_48::M28BYJ_48(uint8_t pin1, uint8_t pin2, uint8_t pin3, uint8_t pin4, uint8_t gearClearance, MotorTypes type) {
   this->pin1 = pin1;
   this->pin2 = pin2;
   this->pin3 = pin3;
   this->pin4 = pin4;
   this->clearance = gearClearance;

   switch (type) {
   case MotorTypes::Simple:
      setPin = &M28BYJ_48::setSinglePin;
      break;
   case MotorTypes::Strong:
      setPin = &M28BYJ_48::setDoublePin;
      break;
   default:
      break;
   }
}

void M28BYJ_48::begin() {
   digitalWrite(pin1, 0);
   digitalWrite(pin2, 0);
   digitalWrite(pin3, 0);
   digitalWrite(pin4, 0);
   pinMode(pin1, OUTPUT);
   pinMode(pin2, OUTPUT);
   pinMode(pin3, OUTPUT);
   pinMode(pin4, OUTPUT);
   (this->*setPin)(stepIndex);
   motorTimer = timerBegin(1'000'000);
   timerAttachInterruptArg(motorTimer, onMotorTimer, this);
   isMotorInitialized = true;
}
void M28BYJ_48::end() {
   pinMode(pin1, INPUT);
   pinMode(pin2, INPUT);
   pinMode(pin3, INPUT);
   pinMode(pin4, INPUT);
   timerDetachInterrupt(motorTimer);
   timerEnd(motorTimer);
   isMotorInitialized = false;
}
void M28BYJ_48::setDirection(Direction dir) {
   if (dir != this->dir) {
      // Apply clearance steps when changing direction
      for (uint8_t i = 0; i < clearance; i++) {
         internalStep(dir);
         delay(50); // Delay to allow motor to step
      }
      Serial.println("Direction changed, applied clearance steps");
   }
   this->dir = dir;
}

void M28BYJ_48::setGearClearance(uint8_t clearance){
   this->clearance = clearance;
   Serial.print("Gear clearance set to ");
   Serial.println(clearance);
}

void M28BYJ_48::internalStep(Direction dir) {
   switch (dir) {
   case Direction::CW:
      internalStepCW();
      break;
   case Direction::CCW:
      internalStepCCW();
      break;
   default:
      break;
   }
}

void M28BYJ_48::step(Direction dir) {
   if (isTimerRunning) {
      stop();
   }
   setDirection(dir);
   internalStep(dir);
}

void M28BYJ_48::internalStepCW() {
   stepIndex = (stepIndex + 1) % 4;
   (this->*setPin)(stepIndex);
}

void M28BYJ_48::internalStepCCW() {
   stepIndex = (stepIndex + 3) % 4;
   (this->*setPin)(stepIndex);
}

void ARDUINO_ISR_ATTR onMotorTimer(void* arg) {
   M28BYJ_48* motor = static_cast<M28BYJ_48*>(arg);
   if (motor->remainingSteps < 0) {
      motor->internalStep(motor->dir);
      return; // unlimited steps
   }

   if (motor->remainingSteps > 0) {
      motor->remainingSteps--;
      motor->internalStep(motor->dir);
      return;
   }

   motor->stop();
}

void M28BYJ_48::internalRun(Direction dir, int steps, uint16_t stepsPerSecond) {
   if (isTimerRunning) {
      stop();
   }
   setDirection(dir);
   timerStop(motorTimer);
   timerAlarm(motorTimer, 1'000'000 / stepsPerSecond, true, 0);
   isTimerRunning = true;
   remainingSteps = steps;
   timerStart(motorTimer);
}

void M28BYJ_48::run(Direction dir, uint16_t stepsPerSecond) {
   internalRun(dir, -1, stepsPerSecond);
}

void M28BYJ_48::run(Direction dir, int steps, uint16_t stepsPerSecond) {
   internalRun(dir, steps, stepsPerSecond);
}

void M28BYJ_48::stop() {
   timerStop(motorTimer);
   isTimerRunning = false;
}

void M28BYJ_48::setSinglePin(uint8_t stepIndex) const {
   switch (stepIndex) {
   case 0:
      digitalWrite(pin1, HIGH);
      digitalWrite(pin2, LOW);
      digitalWrite(pin3, LOW);
      digitalWrite(pin4, LOW);
      break;
   case 1:
      digitalWrite(pin1, LOW);
      digitalWrite(pin2, HIGH);
      digitalWrite(pin3, LOW);
      digitalWrite(pin4, LOW);
      break;
   case 2:
      digitalWrite(pin1, LOW);
      digitalWrite(pin2, LOW);
      digitalWrite(pin3, HIGH);
      digitalWrite(pin4, LOW);
      break;
   case 3:
      digitalWrite(pin1, LOW);
      digitalWrite(pin2, LOW);
      digitalWrite(pin3, LOW);
      digitalWrite(pin4, HIGH);
      break;
   }
}

void M28BYJ_48::setDoublePin(uint8_t stepIndex) const {
   switch (stepIndex) {
   case 0:
      digitalWrite(pin1, HIGH);
      digitalWrite(pin2, HIGH);
      digitalWrite(pin3, LOW);
      digitalWrite(pin4, LOW);
      break;
   case 1:
      digitalWrite(pin1, LOW);
      digitalWrite(pin2, HIGH);
      digitalWrite(pin3, HIGH);
      digitalWrite(pin4, LOW);
      break;
   case 2:
      digitalWrite(pin1, LOW);
      digitalWrite(pin2, LOW);
      digitalWrite(pin3, HIGH);
      digitalWrite(pin4, HIGH);
      break;
   case 3:
      digitalWrite(pin1, HIGH);
      digitalWrite(pin2, LOW);
      digitalWrite(pin3, LOW);
      digitalWrite(pin4, HIGH);
      break;
   }
}




